//*************************************************************************************************
//
//	Description:
//		Water.fx - Water object shader for The Race. Supports vertex colours, Blinn-style diffuse
//		and specular lighting, shininess maps, tangent space normal maps, multiple light sources
//		(of 3 different types), shadow buffers.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     04/05/2006  0.1           Created
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#define MAX_ENVMAP_BIAS	6.0f



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//
#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
>;

// Second UV channel
int texcoord3 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 3;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
> = 0;

// Third UV channel
int texcoord4 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 4;
	int MapChannel = 3;
	int RuntimeTexcoord = 2;
	bool export = false;
> = 0;

#endif


//
// Simulation constants
//

float currentTime : TIME
<
> = 0.0f;

float normal1USpeed
<
  string UIName = "N1 U Speed";
  float UIMin = -1024.0f;
  float UIMax = 1024.0f;
  bool appEdit = true;
> = 0.0f;

float normal1VSpeed
<
  string UIName = "N1 V Speed";
  float UIMin = -1024.0f;
  float UIMax = 1024.0f;
  bool appEdit = true;
> = 0.0f;

float normal2USpeed
<
  string UIName = "N2 U Speed";
  float UIMin = -1024.0f;
  float UIMax = 1024.0f;
  bool appEdit = true;
> = 0.0f;

float normal2VSpeed
<
  string UIName = "N2 V Speed";
  float UIMin = -1024.0f;
  float UIMax = 1024.0f;
  bool appEdit = true;
> = 0.0f;




//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

texture specularTexture : TEXTURE
<
	string UIName = "Specular Tex {UV1}";
	bool appEdit = true;
>;

texture normalTexture1 : TEXTURE
<
	string UIName = "Normal Tex 1 {UV2}";
	bool appEdit = true;
>;

texture normalTexture2 : TEXTURE
<
	string UIName = "Normal Tex 2 {UV3}";
	bool appEdit = true;
>;

texture environmentTexture : TEXTURE					// Environment map (RGB)
<
	string UIName = "Env Texture";
	string Type = "CUBE";
	bool appEdit = true;
>;

float fresnelFactor
<
	string UIName = "Fresnel Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 0.6f;

float minSpecPower
<
	string UIName = "Min Spec Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float maxSpecPower
<
	string UIName = "Max Spec Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 128.0f;

float globalSpecularFactor
<
	string UIName = "Specular Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

float globalEMapFactor
<
	string UIName = "EMap Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;



//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS

// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };


//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D normalMap1 : SAMPLER
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture1"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture1 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


sampler2D normalMap2 : SAMPLER
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture2"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture2 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};



//
// Functions
//


//
// This seems to work almost as well as the full-on "complicated fresnel"
//
float CalculateFresnel( float3 _eye, float3 _normal, float rindexRatio )
{
	// Note: compute R0 on the CPU and provide as a
	// constant; it is more efficient than computing R0 in
	// the vertex shader. R0 is:
	// float const R0 = pow(1.0-refractionIndexRatio, 2.0)
	// / pow(1.0+refractionIndexRatio, 2.0);
	// light and normal are assumed to be normalized
	float R0 = pow( 1.0f - rindexRatio, 2.0f) / pow ( 1.0f + rindexRatio, 2.0f );

	return R0 + ( 1.0f - R0 ) * pow( 1.0f - saturate( dot( -_eye, _normal ) ), 5.0f );
}



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float3 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 0 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD2;													// Vertex alpha
	float2 texCoord1 : TEXCOORD3;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
	float2 texCoord2 : TEXCOORD4;													// UV channel 2 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
#else
	float4 colour   : COLOR0;																// Vertex colour
	float2 texCoord0 : TEXCOORD0;												// UV channel 0 texture coord
	float2 texCoord1 : TEXCOORD1;											// UV channel 1 texture coord
	float2 texCoord2 : TEXCOORD2;											// UV channel 2 texture coord
#endif
	float3 normal   : NORMAL;															// Object space normal
	float3 tangent : TANGENT;														// Object space tangent
	float3 binormal : BINORMAL;														// Object space normal
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;												// View-coords position
	float4 colour			: TEXCOORD6;													// Vertex colour
	float4 texCoord01 : TEXCOORD0;											// UV coords for texture channels 0 and 1
	float2 texCoord2 : TEXCOORD1;												// UV coords for texture channel 2
	float3 normal			: TEXCOORD2;											// Normal vector (world space)
	float4 eye				 : TEXCOORD3;										// Eye vector (world space)
	float3 tangent		 : TEXCOORD4;											// Tangent vector (world space)
	float3 binormal		 : TEXCOORD5;											// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT WaterVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.texCoord01.xy = _input.texCoord0;

#if defined(  _3DSMAX_ )
	// Calculate scrolling offsets for normal map layers 1 and 2
	float2 normal1Vector = float2( normal1USpeed * 0.1f, normal1VSpeed * 0.1f );
	float2 normal2Vector = float2( normal2USpeed * 0.1f, normal2VSpeed * 0.1f );
#else
	// Calculate scrolling offsets for normal map layers 1 and 2
	float2 normal1Vector = float2( round( normal1USpeed * 256.0f ) / 256.0f, round( normal1VSpeed * 256.0f ) / 256.0f );
	float2 normal2Vector = float2( round( normal2USpeed * 256.0f ) / 256.0f, round( normal2VSpeed * 256.0f ) / 256.0f );
#endif

	_output.texCoord01.zw = _input.texCoord1 + ( normal1Vector * currentTime );
	_output.texCoord2 = _input.texCoord2 + ( normal2Vector * currentTime );

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0.0f);

	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.normal	 = normal;
	_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
	_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

struct PSINPUT
{
	float4 colour			: TEXCOORD6;											// Vertex colour
	float4 texCoord01	: TEXCOORD0;									// UV coords for texture channels 0 and 1
	float2 texCoord2	: TEXCOORD1;									// UV coords for texture channel 2
	float3 normal			: TEXCOORD2;									// Normal vector (world space) + fresnel coefficient
	float4 eye				: TEXCOORD3;									// Eye vector (world space)
	float3 tangent		: TEXCOORD4;									// Tangent vector (world space)
	float3 binormal		: TEXCOORD5;									// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
};

#else

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD6;											// Vertex colour
	float4 texCoord01 : TEXCOORD0;									// UV coords for texture channels 0 and 1
	float2 texCoord2	: TEXCOORD1;									// UV coords for texture channel 2
	float3 normal			: TEXCOORD2_centroid;					// Normal vector (world space)
	float4 eye				: TEXCOORD3_centroid;					// Eye vector (world space)
	float3 tangent		: TEXCOORD4_centroid;					// Tangent vector (world space)
	float3 binormal		: TEXCOORD5_centroid;					// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 7 )
	DECLARE_SHADOW_PS_INPUTS
};

#endif	// defined( _3DSMAX_ )


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT WaterFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	float globalSpecularFactorValue = globalSpecularFactor;
	float minSpecPowerValue = minSpecPower;
	float maxSpecPowerValue = maxSpecPower;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord01.xy );
	float4 specularTexColour = tex2D( specularMap, _input.texCoord01.xy );

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

  // Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Normalise the input tangent and binormal vectors
	float3 tangent = normalize( _input.tangent );
	float3 binormal = normalize( _input.binormal );

	// Fetch and decode the map normal
	float4 normalMapColour1 = tex2D( normalMap1, _input.texCoord01.zw );
	float4 normalMapColour2 = tex2D( normalMap2, _input.texCoord2 );
	float3 normalFromMap1 = ( normalMapColour1.rgb * 2.0f ) - 1.0f;
	float3 normalFromMap2 = ( normalMapColour2.rgb * 2.0f ) - 1.0f;

	float3 combinedNormal = normalize( normalFromMap1 + normalFromMap2 );

	// Perturb the tangent space normal by the normal map
	normal = ( TSnormal * combinedNormal.z ) + ( combinedNormal.x * binormal ) + ( combinedNormal.y * tangent );
	normal = normalize( normal );

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	//
	// Calculate the reflection and transmission fresnel factors
	//
	float fresnelCoefficient = CalculateFresnel( -eye, normal, fresnelFactor );
	float transmissionCoefficient = sqrt( fresnelCoefficient );

	// Calculate the envmap space reflection vector
	float3 envMapSpaceNormal = normal;
	float3 envMapSpaceEye = -eye;
	float3 reflectionVector = reflect( envMapSpaceEye, envMapSpaceNormal );

	// Fetch the environment map colour using the world coords vector
#ifdef _3DSMAX_
	float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
	float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

	// Calculate envmap colour and add to diffuse
	accumulatedColour += specularTexColour * environmentTexColour * globalEMapFactor * fresnelCoefficient;

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour, _input.eye.xyz )

	accumulatedColour.w = diffuseTexColour.w * transmissionCoefficient;
	_output.Colour = CalculateOutputPixel( accumulatedColour );

	return _output;
}


//
// Low Detail Shaders
//


struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD1;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
};


VSOUTPUT_LD WaterLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord0;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// get normal in world space and do lighting
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	
	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	// keep alpha value from the vertex
#if defined( _3DSMAX_ )
	_output.colour.a = _input.alpha.r;
#else
	_output.colour.a = _input.colour.a;
#endif

	return _output;
}


PSOUTPUT WaterLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;
	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateLowDetailOutputPixel( accumulatedColour );

	return _output;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Water
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Water";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx WaterVertexShader();
		PixelShader = compile sce_fp_rsx WaterFragmentShader();
#else
		VertexShader = compile vs_3_0 WaterVertexShader();
		PixelShader = compile ps_3_0 WaterFragmentShader();
#endif
	}
}



technique Water_Translucent
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "Water";
	int    normalDeferredID			= 2;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 WaterVertexShader();
		PixelShader = compile ps_3_0 WaterFragmentShader();
#endif
	}
}

